/** @file   progressindicator.cpp
 * @brief   Implementation of ProgressIndicator class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:40 $
 * @author  Tomi Lamminsaari
 */


#include "progressindicator.h"
#include <allegro.h>
using eng2d::Color;
using eng2d::Gradient;


namespace eng2d {


//********************************************************************
//                                                                   *
//      Static members and constants                                 *
//                                                                   *
//********************************************************************

Color ProgressIndicator::borderColor(122,52,7);
Gradient ProgressIndicator::fillGradient( Color(0,0,0), Color(200,0,0) );


//********************************************************************
//                                                                   *
//      Constructors, destructor and operators                       *
//                                                                   *
//********************************************************************

/** Constructs new ProgressIndicator with given number of steps
 */
ProgressIndicator::ProgressIndicator(int totalSteps) :
  m_totalSteps( totalSteps ),
  m_currentStep( 0 ),
  m_posX(0),
  m_posY(0),
  m_width(320),
  m_height(20),
  m_pTarget( screen )
{
}



/** Constructs new ProgressIndicator for given number of steps and that
 * will be drawn on given bitmap.
 */
ProgressIndicator::ProgressIndicator( int totalSteps, BITMAP* pTarget ):
  m_totalSteps( totalSteps ),
  m_currentStep( 0 ),
  m_posX(0),
  m_posY(0),
  m_width(320),
  m_height(20),
  m_pTarget( pTarget )
{
}



/** Copy constructor
 */
ProgressIndicator::ProgressIndicator( const ProgressIndicator& rO ) :
  m_totalSteps( rO.m_totalSteps ),
  m_currentStep( rO.m_currentStep ),
  m_posX( rO.m_posX ),
  m_posY( rO.m_posY ),
  m_width( rO.m_width ),
  m_height( rO.m_height ),
  m_pTarget( rO.m_pTarget )
{
}



/** Destructor
 */
ProgressIndicator::~ProgressIndicator()
{
}



/** Operator =
 */
ProgressIndicator& ProgressIndicator::operator = ( const ProgressIndicator& rO )
{
  m_totalSteps = rO.m_totalSteps;
  m_currentStep = rO.m_currentStep;
  m_posX = rO.m_posX;
  m_posY = rO.m_posY;
  m_width = rO.m_width;
  m_height = rO.m_height;
  m_pTarget = rO.m_pTarget;
  return *this;
}



//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************

/** Sets the current step
 */
void ProgressIndicator::setStep(int step)
{
  m_currentStep = step;
}



/** Increments the current step by 1
 */
void ProgressIndicator::advance()
{
  m_currentStep += 1;
}



/** Draws this indicator to the given target bitmap
 */
void ProgressIndicator::redraw()
{
  int rx = m_posX + m_width;
  int ry = m_posY + m_height;
  
  float percentsDone = static_cast<float>( m_currentStep ) /
                       static_cast<float>( m_totalSteps );

  float offsetX = percentsDone * static_cast<float>( m_width );
  int barX = static_cast<int>( offsetX ) + m_posX;
  
  // Draw the outlined rectangle
  rect( m_pTarget, m_posX,m_posY, rx,ry, borderColor.allC() );
  
  // Draw progress bar
  for (int i=m_posX+1; i < barX; i++) {
    Color c = fillGradient.getColorAt( i, m_posX, barX );
    vline( m_pTarget, i, m_posY+1, ry-1, c.allC() );
  }
}



/** Sets the position of this indicator
 */
void ProgressIndicator::position(int x, int y)
{
  m_posX = x;
  m_posY = y;
}



/** Sets the size of this indicator in pixels
 */
void ProgressIndicator::size(int w, int h)
{
  m_width = w;
  m_height = h;
}



//********************************************************************
//                                                                   *
//      Public GET - methods                                         *
//                                                                   *
//********************************************************************

/** Returns the current progress step
 */
int ProgressIndicator::currentStep() const
{
  return m_currentStep;
}



/** Returns the total number of steps
 */
int ProgressIndicator::totalSteps() const
{
  return m_totalSteps;
}

} // end of namespace
